# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

import bpy
from bpy.props import IntProperty
from .packing import pack_textures

class OBJECT_OT_LilyTexturePacker(bpy.types.Operator):
    """Pack many textures into one single image"""
    bl_idname = "object.lily_texture_packer"
    bl_label = "Lily Texture Packer"
    bl_options = {"REGISTER", "UNDO"}

    spacing: IntProperty(
        name="Spacing",
        description="Number of pixels to leave unused between packed objects",
        default=0,
    )

    max_texture_size: IntProperty(
        name = "Max Texture Size",
        description="If not zero, maximum width and height of packed images",
        default=0,
    )

    @classmethod
    def getObjects(cls, context):
        return [obj for obj in context.selected_objects if obj.type == 'MESH']

    @classmethod
    def poll(cls, context):
        return len(cls.getObjects(context)) > 0

    def execute(self, context):
        pack_textures(self.getObjects(context), spacing=self.spacing, max_texture_size=self.max_texture_size)
        return {'FINISHED'}

def menu_func_import(self, context):
    self.layout.operator(OBJECT_OT_LilyTexturePacker.bl_idname)

def register():
    bpy.utils.register_class(OBJECT_OT_LilyTexturePacker)
    bpy.types.VIEW3D_MT_object.append(menu_func_import)

def unregister():
    bpy.utils.unregister_class(OBJECT_OT_LilyTexturePacker)
    bpy.types.VIEW3D_MT_object.remove(menu_func_import)
